/*
 *
 * This original maze generation program created
 *  entirely from scratch by Danny Brewer.
 * Program started on 4/20/1998.
 * Heavily revised and improved 3/1/2003.
 * Copyright 1998 Danny Brewer.  All rights reserved.
 * 
 * Copyright 2003 Danny Brewer
 * Anyone may run this code.
 * If you wish to modify or distribute this code, then
 *  you are granted a license to do so only under the terms
 *  of the Gnu Lesser General Public License.
 * See:  http://www.gnu.org/licenses/lgpl.html
 */
/*
	----+---+---------------------------------------------------+
	  * |   |     * * * * * * *                                 |
	| * |   +---- * ----+---- * +---+   +-----------------------+
	| * * * | * * *     |     * |   |   |                       |
	+---+ * | * +---+---+   | * |   +---+---+---+   +-----------+
	|   | * | * |   |   |   | * * * * * * * |   |   |           |
	|   | * | * |   |   +---+-------+---- * |   |   |   +-------+
	|   | * | * | * * * * * * * * * |     * |       |   |       |
	|   | * | * | * ----+-------- * +---- * |   ----+   +----   |
	| * * * | * | *     | * * * * * |     * * * * * * * * * * * |
	| * ----+ * | * +---+ * |   +---+   +-------+-----------+ * |
	| * * * | * | * |     * |   |       |       | * * *     | * |
	+---- * | * | * +---- * +---+----   |   ----+ * | * ----+ * |
	| * * * | * * * | * * *     |       |       | * | * * * * * |
	| * ----+-------+ * +---+---+---+   |   ----+ * +-----------+
	| * * * * * * * * * |   |       |           | * * * * *     |
	|   +-----------+---+   +----   +----   ----+-------- * +---+
	|   |           |                           | * * * * * |   |
	|   |   ----+   +---------------------------+ * --------+   |
	|           |                               | * * * * * * *  
	+-----------+-------------------------------+----------------
*/



package nom.DannyBrewer.recreation.squareMaze;


abstract public class SquareMazePrinter {
	private SquareMaze maze = null;
	
    //----------------------------------------------------------------------
    //  Constructor
    //----------------------------------------------------------------------
	
	public SquareMazePrinter( SquareMaze maze ) {
		this.maze = maze;
	}


    //----------------------------------------------------------------------
    //  Public API
    //----------------------------------------------------------------------
	
	private int textLinesPerCell = 1;
	public int getTextLinesPerCell() { return textLinesPerCell; }
	public void setTextLinesPerCell( int lines ) { textLinesPerCell = lines; }
	
	// Default output is System.out.
	private java.io.PrintStream output = System.out;
	public java.io.PrintStream getOutputStream() { return output; }
	public void setOutputStream( java.io.PrintStream out ) { output = out; }
	
	
	public void printMaze() {
		printMaze( false );
	}
	
	public void printMaze( boolean showSolution ) {
		boolean showWall, showWallAsSolution;
		int rows = maze.getRows();
		int cols = maze.getCols();
		
		for( int row = 0;  row <= rows;  ++row ) {
			
			//--------------------
			// Print the top of the cells.
			for( int col = 0; col <= cols; ++col ) {
				
				// Decide which four parts of a cross to print.
				boolean topFlag = false, leftFlag = false,
						bottomFlag = false, rightFlag = false;
				
				if( hasWall( row-1, col-1, SquareMaze.DIRECTION_RIGHT ) ) topFlag = true;
				if( hasWall( row-1, col-1, SquareMaze.DIRECTION_BOTTOM ) ) leftFlag = true;
				if( hasWall( row, col-1, SquareMaze.DIRECTION_RIGHT ) ) bottomFlag = true;
				if( hasWall( row, col-1, SquareMaze.DIRECTION_TOP ) ) leftFlag = true;
				if( hasWall( row-1, col, SquareMaze.DIRECTION_LEFT ) ) topFlag = true;
				if( hasWall( row-1, col, SquareMaze.DIRECTION_BOTTOM ) ) rightFlag = true;
				if( hasWall( row, col, SquareMaze.DIRECTION_LEFT ) ) bottomFlag = true;
				if( hasWall( row, col, SquareMaze.DIRECTION_TOP ) ) rightFlag = true;
				
				printCross( topFlag, leftFlag, bottomFlag, rightFlag );
				
				if( col < cols ) {
					showWall = false;
					if( hasWall( row-1, col, SquareMaze.DIRECTION_BOTTOM ) ) showWall = true;
					if( hasWall( row, col, SquareMaze.DIRECTION_TOP ) ) showWall = true;

					showWallAsSolution = false;
					if( showSolution ) {
						// If both adjacent cells are solutions, then the wall should show as a solution.
						if( isSolutionCell( row-1, col )  &&  isSolutionCell( row, col ) ) {
							showWallAsSolution = true;
						}

						// If either cell is a solution and there is no wall...
						else if( (! showWall)  &&  (isSolutionCell( row-1, col )  ||  isSolutionCell( row, col )) ) {
							// ...and we're printing the extreme top or bottom edge, then print wall as a solution.
							if( row == 0  ||  row == rows ) {
								showWallAsSolution = true;
							}
						}
					}

					printHorizWall( showWall, showWallAsSolution );
				}
			}
			println();
			// Finished printing tops of cells.
			//--------------------
			
			if( row < rows ) {
				//--------------------
				// Now print contents of cells.
				// Print each line potentially multiple times = textLinesPerCell.
				for( int rowLine = 0;  rowLine < textLinesPerCell;  ++rowLine ) {
					for( int col = 0;  col <= cols;  ++col ) {
						showWall = false;
						if( hasWall( row, col-1, SquareMaze.DIRECTION_RIGHT ) ) showWall = true;
						if( hasWall( row, col, SquareMaze.DIRECTION_LEFT ) ) showWall = true;
						
						showWallAsSolution = false;
						if( showSolution ) {
							// If both adjacent cells are solutions, then the wall should show as a solution.
							if( isSolutionCell( row, col-1 )  &&  isSolutionCell( row, col ) ) {
								showWallAsSolution = true;
							}
							
							// If either cell is a solution and there is no wall...
							else if( (! showWall)  &&  (isSolutionCell( row, col-1 )  ||  isSolutionCell( row, col )) ) {
								// ...and we're printing the extreme left or right edge, then print wall as a solution.
								if( col == 0  ||  col == cols ) {
									showWallAsSolution = true;
								}
							}
						}
						
						printVertWall( showWall, showWallAsSolution );
						
						if( col < cols ) {
							printInnerCell( showSolution  &&  isSolutionCell( row, col ) );
						}
					}
					println();
				}
				// Finished printing contents of cells
				//--------------------
			}
		}
	}
	

	
    //----------------------------------------------------------------------
    //  Subclass API
    //----------------------------------------------------------------------
	
	protected void println() {
		output.println();
	}
	
	protected void print( String s ) {
		output.print( s );
	}
	
	
	//----------------------------------------------------------------------
    //  SquareMazePrinter interface
    //----------------------------------------------------------------------
	
	// Print the wall that borders the top or bottom of a cell.
	abstract protected void printHorizWall( boolean wall, boolean solution );
	
	// Print the wall that borders the left or right of a cell.
	abstract protected void printVertWall( boolean wall, boolean solution );
	
	// Print one character at the intersection of vertical and horizontal walls.
	// The four parameters indicate which parts of the cross should be printed.
	// The following implementation is for generic ASCII, but an implementation could
	//  be substituted that would use the line, box corner, T, upside down T, etc.
	//  characters of an IBM PC.
	abstract protected void printCross( boolean top, boolean left, boolean bottom, boolean right );
	
	// Print the contents of a cell inside its walls.
	abstract protected void printInnerCell( boolean solution );

		
	//----------------------------------------------------------------------
    //  Internal Support
    //----------------------------------------------------------------------
	
	protected boolean hasWall( int row, int col, int direction ) {
		SquareMaze.MazeCell cell = maze.getMazeCell( row, col );
		if( cell != null ) {
			return cell.hasWall( direction );
		}
		return false;
	}
	
	protected boolean isSolutionCell( int row, int col ) {
		SquareMaze.MazeCell cell = maze.getMazeCell( row, col );
		if( cell != null ) {
			return cell.isOnSolutionPath();
		}
		return false;
	}

	
} // class SquareMazePrinter



